/*+ TsipParser.cpp
 *
 ******************************************************************************
 *
 *                        Trimble Navigation Limited
 *                           645 North Mary Avenue
 *                              P.O. Box 3642
 *                         Sunnyvale, CA 94088-3642
 *
 ******************************************************************************
 *
 *    Copyright  2005 Trimble Navigation Ltd.
 *    All Rights Reserved
 *
 ******************************************************************************
 *
 * Description:
 *    This file implements the CTsipParser class.
 *            
 * Revision History:
 *    05-18-2005    Mike Priven
 *                  Written
 *
 * Notes:
 *
-*/

/*---------------------------------------------------------------------------*\
 |                         I N C L U D E   F I L E S
\*---------------------------------------------------------------------------*/
#include "stdafx.h"
#include "TsipParser.h"
#include "CommPort.h"
#include <math.h>


/*---------------------------------------------------------------------------*\
 |                 T S I P   P R O C E S S O R   R O U T I N E S 
\*---------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
Function:       IsStartOfTsip

Description:    Determines if the supplied byte could be the start of a new 
				TSIP packet, i.e. if it is a DLE. 

Parameters:     ucByte		 - a byte to test.

Return Value:   True if the byte could be the start of a new TSIP packet, 
				false otherwise. 
-----------------------------------------------------------------------------*/
bool    CTsipParser::IsStartOfTsip(U8 ucByte)
{
	return DLE == ucByte; 
}


/*-----------------------------------------------------------------------------
Function:       Reset

Description:    Resets the parsing state machine to its initial state. 

                NOTE: This function must be called prior to the first call of
				ReceiveByte when receiving a new TSIP packet. 

Parameters:     None

Return Value:   None 
-----------------------------------------------------------------------------*/
void CTsipParser::Reset()
{
	m_nParseState = MSG_IN_COMPLETE; 
}


/*-----------------------------------------------------------------------------
Function:       ReceiveByte

Description:    Receives a complete TSIP packet one byte at a time. Call the
				function repeatedly until it indicates that a full packet has
				been received (see Return Value below). 

                The entire packet (the starting DLE, packet ID, packet data, 
                and trailing DLE and ETX) is placed in the buffer ucPkt. The 
                entire packet size is stored in pPktLen.

Parameters:     ucByte		 - The next byte to be added to the message. 
                ucPkt        - a memory buffer where the entire TSIP packet 
                               will be stored.
                pPktLen      - a pointer to a variable to be updated with the
                               packet size (which includes the packet header
                               and trailing bytes).

Return Value:   The internal parsing state:

				MSG_IN_ERROR: There was an error receiving the message. Reset()
				should be called and the input stream re-synched using 
				IsStartOfTsip().

				MSG_IN_COMPLETE: A complete message has been received in ucPkt
				and can be parsed by ParsePkt(). 

				Any other value: More bytes are needed to complete the packet, 
				keep feeding them in. 
-----------------------------------------------------------------------------*/
int CTsipParser::ReceiveByte (unsigned char ucByte, 
                              unsigned char ucPkt[], 
                              int           *pPktLen)
{

	// The TSIP packet is received in a local state machine.
	switch (m_nParseState) 
	{
	case MSG_IN_ERROR:		// Start over in case of error
	case MSG_IN_COMPLETE:               
		// This is the initial state in which we look for the start
		// of the TSIP packet. We can also end up in this state if
		// we received too many data bytes from the serial port but
		// did not find a valid TSIP packet in that data stream.
		// 
		// While in this state, we look for a DLE character. If we
		// are in this state and the DLE is received, we initialize
		// the packet buffer and transition to the next state.
		if (ucByte == DLE) 
		{
			m_nParseState    = TSIP_DLE;
			*pPktLen          = 0;
			ucPkt[(*pPktLen)++] = ucByte;
		}
		break;

	case TSIP_DLE:
		// The parser transitions to this state if a previosly
		// received character was DLE. Receipt of this character
		// indicates that we may be in one of three situations:
		//
		//     Case 1: DLE ETX  = end of a TSIP packet
		//     Case 2: DLE <id> = start of a TSIP packet <id>
		//     Case 3: DLE DLE  = a DLE byte inside the packet
		//                        (stuffed DLE byte which is a part
		//                        of the TSIP data)
		// 
		// To distinguish amongst these, we look at the next
		// character (currently read into ucByte). 
		// 
		// If the next character is ETX (Case 1), it's the end the 
		// TSIP packet. At this point, we either have a complete TSIP
		// packet in ucPkt or an empty packet. If we have a complete
		// packet, return it to the caller. Otherwise, go back
		// to the intial state and look for a valid packet again.
		//
		// If the next character is anything other than ETX, we
		// add the character to the packet buffer and transition to
		// next state to distinguish between Cases 2 and 3.
		if (ucByte == ETX) 
		{
			if (*pPktLen > 1)
			{
				ucPkt[(*pPktLen)++] = DLE;
				ucPkt[(*pPktLen)++] = ETX;

				m_nParseState = MSG_IN_COMPLETE;
			}
			else
			{
				m_nParseState = MSG_IN_ERROR;
			}
		}
		else  
		{
			m_nParseState = TSIP_IN_PARTIAL;
			ucPkt[(*pPktLen)++] = ucByte;
		}
		break;

	case TSIP_IN_PARTIAL:
		// The parser is in this state if a previous character was
		// a part of the TSIP data. As noted above, a DLE character
		// can be a part of the TSIP data in which case another DLE
		// character is present in the data stream. So, here we look 
		// at the next character in the stream (currently loaded in 
		// ucByte). If it is a DLE character, we just encountered
		// a stuffed DLE byte. In that case, we ignore this byte
		// and go back to the TSIP_DLE state. That way, we will log
		// only one DLE byte which was a part of the TSIP data.
		//
		// All other non-DLE characters are placed in the TSIP packet
		// buffere.
		if (ucByte == DLE) 
		{
			m_nParseState = TSIP_DLE;
		}
		else 
		{
			ucPkt[(*pPktLen)++] = ucByte;
		}
		break;

	default:
		// We should never be in this state. This is just for a good
		// programming style.
		m_nParseState = MSG_IN_ERROR;
		break;
	}

	// We get to this point after reading each byte from the serial port.
	// Because it is not the end of the valid TSIP packet yet, we check for 
	// buffer overflow. If it overflows we assume something went wrong 
	// with the end of message character(s) since no input message should
	// be bigger than MAX_TSIP_PKT_LEN. We ignore this message and wait till 
	// the next message starts.                                    
	if (*pPktLen >= MAX_TSIP_PKT_LEN) 
	{
		m_nParseState = MSG_IN_ERROR;
		*pPktLen = 0;
	}

	return m_nParseState; 
}

/*-----------------------------------------------------------------------------
Function:       ParsePkt

Description:    This function extracts the data values from a TSIP packet
                and returns an ASCII-formatted string with the values.

Parameters:     ucPkt   - a memory buffer with the entire TSIP packet 
                nPktLen - size of the packet (including the header and 
                          trailing bytes).

Return Value:   A string with TSIP data values extracted and formatted as 
                ASCII text.
-----------------------------------------------------------------------------*/
CString CTsipParser::ParsePkt (unsigned char ucPkt[], int nPktLen)
{
    // The CTsipParser object is non-rentrant, and ParsePkt can only
    // be called from one thread because we use a global string to
    // store the parsed TSIP data.
    m_str = "";

    // ucPkt contains the entire TSIP packet including the leading
    // DLE (0x10) and the trailing DLE and ETX (0x03). 
    
    // Here, based on the TSIP packet ID (indicated by the second
    // byte of the packet buffer, we pass the pointer and size
    // of the actual packet data to an appropriate parser. Note that
    // we only pass the address of the first actual data byte and
    // only the size of data (which excludes the first DLE, packet
    // ID byte, and trailing DLE and ETX.

    switch (ucPkt[1])
    {
        case 0x41: Parse0x41 (&ucPkt[2], nPktLen-4); break;
        case 0x42: Parse0x42 (&ucPkt[2], nPktLen-4); break;
        case 0x43: Parse0x43 (&ucPkt[2], nPktLen-4); break;
        case 0x45: Parse0x45 (&ucPkt[2], nPktLen-4); break;
        case 0x46: Parse0x46 (&ucPkt[2], nPktLen-4); break;
        case 0x4A: Parse0x4A (&ucPkt[2], nPktLen-4); break;
        case 0x4B: Parse0x4B (&ucPkt[2], nPktLen-4); break;
        case 0x55: Parse0x55 (&ucPkt[2], nPktLen-4); break;
        case 0x56: Parse0x56 (&ucPkt[2], nPktLen-4); break;
        case 0x6D: Parse0x6D (&ucPkt[2], nPktLen-4); break;
        case 0x82: Parse0x82 (&ucPkt[2], nPktLen-4); break;
        case 0x83: Parse0x83 (&ucPkt[2], nPktLen-4); break;
        case 0x84: Parse0x84 (&ucPkt[2], nPktLen-4); break;
        case 0x8F: Parse0x8F (&ucPkt[2], nPktLen-4); break;
		default: m_str.Format("Unsupported packet id: 0x%02x", ucPkt[1]); break;
    }

    // Each of the individual packet parsers formats m_str with the parsed
    // TSIP data values represented as an ASCII text string.
    return m_str;
}


/*---------------------------------------------------------------------------*\
 |     I N D I V I D U A L   T S I P   P A C K E T   P A R S E R S
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       Parse0x41

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x41 (unsigned char ucData[], int nLen)
{
    FLT fltTimeOfWeek, fltUtcOffset;
    S16 sWeekNum;

    // Check the length of the data string
    if (nLen != 10)
    {
        return;
    }

    // Extract values from the data string
    fltTimeOfWeek = GetSingle (ucData);
    sWeekNum      = GetShort  (&ucData[4]);
    fltUtcOffset  = GetSingle (&ucData[6]);

    // Format the output string
    m_str.Format ("GPS time:%s GPS week: %d   UTC offset %.1f",
                  ShowTime (fltTimeOfWeek), sWeekNum, fltUtcOffset);
}

/*-----------------------------------------------------------------------------
Function:       Parse0x42

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x42 (unsigned char ucData[], int nLen)
{
    FLT fltEcefPos[3], fltTimeOfFix;

    // Check the length of the data string
    if (nLen != 16) 
    {
        return;
    }

    // Extract values from the data string
    fltEcefPos[0] = GetSingle (ucData);
    fltEcefPos[1] = GetSingle (&ucData[4]);
    fltEcefPos[2] = GetSingle (&ucData[8]);
    fltTimeOfFix  = GetSingle (&ucData[12]);

    // Format the output string
    m_str.Format ("SXYZ:  %15.0f  %15.0f  %15.0f    %s",
                  fltEcefPos[0], fltEcefPos[1], fltEcefPos[2],
                  ShowTime (fltTimeOfFix));
}

/*-----------------------------------------------------------------------------
Function:       Parse0x43

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x43 (unsigned char ucData[], int nLen)
{
    FLT fltEcefVel[3], fltFreqOffset, fltTimeOfFix;

    // Check the length of the data string
    if (nLen != 20) 
    {
        return;
    }

    // Extract values from the data string
    fltEcefVel[0] = GetSingle (ucData);
    fltEcefVel[1] = GetSingle (&ucData[4]);
    fltEcefVel[2] = GetSingle (&ucData[8]);
    fltFreqOffset = GetSingle (&ucData[12]);
    fltTimeOfFix  = GetSingle (&ucData[16]);

    // Format the output string
    m_str.Format ("VelECEF: %11.3f  %11.3f  %11.3f  %12.3f%s",
                  fltEcefVel[0], fltEcefVel[1], fltEcefVel[2], fltFreqOffset,
                  ShowTime (fltTimeOfFix));
}

/*-----------------------------------------------------------------------------
Function:       Parse0x45

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x45 (unsigned char ucData[], int nLen)
{
    U8 ucNavMajor, ucNavMinor, ucNavDay, ucNavMonth, ucNavYear;
    U8 ucDspMajor, ucDspMinor, ucDspDay, ucDspMonth, ucDspYear;

    // Check the length of the data string
    if (nLen != 10)
    {
        return;
    }

    // Extract values from the data string
    ucNavMajor = ucData[0];
    ucNavMinor = ucData[1];
    ucNavDay   = ucData[2];
    ucNavMonth = ucData[3];
    ucNavYear  = ucData[4];
    ucDspMajor = ucData[5];
    ucDspMinor = ucData[6];
    ucDspDay   = ucData[7];
    ucDspMonth = ucData[8];
    ucDspYear  = ucData[9];

    // Format the output string
    m_str.Format ("FW Versions:  Nav Proc %2d.%02d  %02d/%02d/%d  Sig Proc %2d.%02d  %02d/%02d/%d",
                  ucNavMajor, ucNavMinor, ucNavDay, ucNavMonth, ucNavYear+1900,
                  ucDspMajor, ucDspMinor, ucDspDay, ucDspMonth, ucDspYear+1900);
}

/*-----------------------------------------------------------------------------
Function:       Parse0x46

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x46 (unsigned char ucData[], int nLen)
{
    U8 ucStatusCode, ucErrorCode;

    // Check the length of the data string
    if (nLen != 2)
    {
        return;
    }

    // Extract values from the data string
    ucStatusCode = ucData[0];
    ucErrorCode  = ucData[1];

    // These text descriptions are used for formatting below.
    const char *strStatusCode[MAX_SC_MESSAGE] = 
    {
        "Doing position fixes",
        "Don't have GPS time yet",
        "Waiting for almanac collection",
        "DOP too high",
        "No satellites available",
        "Only 1 satellite available",
        "Only 2 satellites available",
        "Only 3 satellites available",
        "No satellites usable",
        "Only 1 satellite usable",
        "Only 2 satellites usable",
        "Only 3 satellites usable",
        "Chosen satellite unusable"
    };

    const char *strErrorCode[MAX_EC_MESSAGE] = 
    {
        "RAM cleared at power-up",
        "Signal processor error",
        "Channel 1 Misalignment",
        "Channel 2 Misalignment",
        "Antenna line open/short",
        "Excessive Osc Freq Bias"
    };

    // Format the output string
    if (ucStatusCode > MAX_SC_MESSAGE)
    {
        m_strTemp.Format("Unknown status code 0x%02X", ucStatusCode);
    }
    else
    {
        m_strTemp.Format("Rcvr status code = 0x%02X (%s)",
                         ucStatusCode, strStatusCode[ucStatusCode]);
    }
    m_str += m_strTemp;

    m_strTemp.Format("\r\n   Receiver health byte       (0x%02X)",
                     ucErrorCode);
    m_str += m_strTemp;

    for (int i=0; i<MAX_EC_MESSAGE; i++) 
    {
        if (ucErrorCode & (1 << i)) 
        {
            m_strTemp.Format("\r\n   %s    (0x%02X)", 
                             strErrorCode[i], (1<<i));
            m_str += m_strTemp;
        }
    }
}

/*-----------------------------------------------------------------------------
Function:       Parse0x4A

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

                TSIP packet 0x4A can have two versions. Depending on the size,
                different data values are extracted.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x4A (unsigned char ucData[], int nLen)
{
    // Check the length of the data string
    if (nLen == 20) 
    {
        Parse0x4ALong (ucData, nLen);
    }
    else if (nLen == 9)
    {
        Parse0x4AShort (ucData, nLen);
    }
}

/*-----------------------------------------------------------------------------
Function:       Parse0x4AShort

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

                Note: this function parses the short version of the TSIP
                packet 0x4A (reference altitude)

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x4AShort (unsigned char ucData[], int nLen)
{
    FLT fltRefAlt, fltReserved;
    U8  ucAltFlag;

    // Check the length of the data string
    if (nLen != 9) 
    {
        return;
    }

    // Extract values from the data string
    fltRefAlt   = GetSingle (ucData);
    fltReserved = GetSingle (&ucData[4]);
    ucAltFlag   = ucData[8];

    // Format the output string
    m_str.Format ("Reference Alt:   %.1f m;    %s", 
                  fltRefAlt, (ucAltFlag ? "ON":"OFF"));
}

/*-----------------------------------------------------------------------------
Function:       Parse0x4ALong

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

                Note: this function parses the longversion of the TSIP
                packet 0x4A (single precision LLA output)

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x4ALong (unsigned char ucData[], int nLen)
{
    FLT fltLat, fltLon, fltAlt, fltClockBias, fltTimeOfFix;
    S16 sLatDeg, sLonDeg;
    DBL dblLatMin, dblLonMin;
    S8  cNorthSouth, cEastWest;

    // Check the length of the data string
    if (nLen != 20) 
    {
        return;
    }

    // Extract values from the data string
    fltLat       = GetSingle (ucData);
    fltLon       = GetSingle (&ucData[4]);
    fltAlt       = GetSingle (&ucData[8]);
    fltClockBias = GetSingle (&ucData[12]);
    fltTimeOfFix = GetSingle (&ucData[16]);

    // Convert from radians to degrees
    fltLat *= (FLT)R2D;
    if (fltLat < 0.0) 
    {
        cNorthSouth = 'S';
        fltLat = -fltLat;
    }
    else
    {
        cNorthSouth = 'N';
    }
    sLatDeg = (S16)fltLat;
    dblLatMin = (fltLat - sLatDeg) * 60.0;

    fltLon *= (FLT)R2D;
    if (fltLon < 0.0) 
    {
        cEastWest = 'W';
        fltLon = -fltLon;
    }
    else
    {
        cEastWest = 'E';
    }
    sLonDeg = (S16)fltLon;
    dblLonMin = (fltLon - sLonDeg) * 60.0;

    // Format the output string
    m_str.Format("SLLA: %4d: %06.3f  %c%5d:%06.3f  %c%10.2f  %12.2f%s",
                 sLatDeg, dblLatMin, cNorthSouth, 
                 sLonDeg, dblLonMin, cEastWest, 
                 fltAlt, fltClockBias, ShowTime(fltTimeOfFix));
}

/*-----------------------------------------------------------------------------
Function:       Parse0x4B

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x4B (unsigned char ucData[], int nLen)
{
    U8 ucMachineID, ucStatus1, ucStatus2;

    // Check the length of the data string
    if (nLen != 3)
    {
        return;
    }

    // Extract values from the data string
    ucMachineID = ucData[0];
    ucStatus1   = ucData[1];
    ucStatus2   = ucData[2];

    // These text descriptions are used for formatting below.
    const char *strStatus1[MAX_AS1_MESSAGE] = 
    {
        "Synthesizer fault         ",
        "No RTC Timeset at power-up",
        "A-to-D converter fault    ",
        "Almanac not complete      "
    };

    // Format the output string
    m_strTemp.Format("Machine/Code ID: %d   Status: (0x%02X) %s",
                     ucMachineID, ucStatus1, 
                     ((ucStatus2 & 0x01) ? "  Superpkts supported" : ""));
    m_str += m_strTemp;

    for (int i=0; i<MAX_AS1_MESSAGE; i++) 
    {
        if (ucStatus1 & (1 << i))           
        {
            m_strTemp.Format("\r\n   %s (0x%02X)", strStatus1[i], (1<<i));
            m_str += m_strTemp;
        }
    }
}

/*-----------------------------------------------------------------------------
Function:       Parse0x55

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x55 (unsigned char ucData[], int nLen)
{
    U8 ucPosOptions, ucVelOptions, ucTimeOptions, ucAuxOptions;

    // Check the length of the data string
    if (nLen != 4) 
    {
        return;
    }

    // Extract values from the data string
    ucPosOptions  = ucData[0];
    ucVelOptions  = ucData[1];
    ucTimeOptions = ucData[2];
    ucAuxOptions  = ucData[3];

    // Format the output string
    m_strTemp.Format ("I/O Options: %X %X %X %X",
                      ucPosOptions, ucVelOptions, ucTimeOptions, ucAuxOptions);
    m_str += m_strTemp;

    if (ucPosOptions & 0x01)
    {
        m_strTemp.Format ("\r\n    ECEF XYZ position output");
        m_str += m_strTemp;
    }

    if (ucPosOptions & 0x02)
    {
        m_strTemp.Format ("\r\n    LLA position output");
        m_str += m_strTemp;
    }

    m_strTemp.Format ((ucPosOptions & 0x04) ?
                         "\r\n    MSL altitude output (Geoid height) " :
                         "\r\n    WGS-84 altitude output");
    m_str += m_strTemp;

    m_strTemp.Format ((ucPosOptions & 0x08) ?
                         "\r\n    MSL altitude input" :
                         "\r\n    WGS-84 altitude input");
    m_str += m_strTemp;

    m_strTemp.Format ((ucPosOptions & 0x10) ?
                         "\r\n    Double precision" :
                         "\r\n    Single precision");
    m_str += m_strTemp;

    if (ucPosOptions & 0x20)
    {
        m_strTemp.Format ("\r\n    All Enabled Superpackets");
        m_str += m_strTemp;
    }

    if (ucVelOptions & 0x01)
    {
        m_strTemp.Format ("\r\n    ECEF XYZ velocity output");
        m_str += m_strTemp;
    }

    if (ucVelOptions & 0x02)
    {
        m_strTemp.Format ("\r\n    ENU velocity output");
        m_str += m_strTemp;
    }

    m_strTemp.Format ((ucTimeOptions & 0x01) ?
                         "\r\n    Time tags in UTC" :
                         "\r\n    Time tags in GPS time");
    m_str += m_strTemp;

    if (ucTimeOptions & 0x02)
    {
        m_strTemp.Format ("\r\n    Fixes delayed to integer seconds");
        m_str += m_strTemp;
    }

    if (ucTimeOptions & 0x04)
    {
        m_strTemp.Format ("\r\n    Fixes sent only on request");
        m_str += m_strTemp;
    }

    if (ucTimeOptions & 0x08)
    {
        m_strTemp.Format ("\r\n    Synchronized measurements");
        m_str += m_strTemp;
    }

    if (ucTimeOptions & 0x10)
    {
        m_strTemp.Format ("\r\n    Minimize measurement propagation");
        m_str += m_strTemp;
    }

    if (ucTimeOptions & 0x20)
    {
        m_strTemp.Format ("\r\n    PPS output at all time");
        m_str += m_strTemp; 
    }
    else
    {
        m_strTemp.Format ("\r\n    PPS output during fixes");
        m_str += m_strTemp;
    }

    if (ucAuxOptions & 0x01)
    {
        m_strTemp.Format ("\r\n    Raw measurement output");
        m_str += m_strTemp;
    }

    if (ucAuxOptions & 0x02)
    {
        m_strTemp.Format ("\r\n    Code-phase smoothed before output");
        m_str += m_strTemp;
    }

    if (ucAuxOptions & 0x04)
    {
        m_strTemp.Format ("\r\n    Additional fix status");
        m_str += m_strTemp;
    }

    m_strTemp.Format ("\r\n    Signal Strength Output as %s",
                      ((ucAuxOptions & 0x08) ? "dBHz" : "AMU"));
    m_str += m_strTemp;
}

/*-----------------------------------------------------------------------------
Function:       Parse0x56

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x56 (unsigned char ucData[], int nLen)
{
    FLT fltEnuVel[3], fltFreqOffset, fltTimeOfFix;

    // Check the length of the data string
    if (nLen != 20) 
    {
        return;
    }

    // Extract values from the data string
    fltEnuVel[0]  = GetSingle (ucData);
    fltEnuVel[1]  = GetSingle (&ucData[4]);
    fltEnuVel[2]  = GetSingle (&ucData[8]);
    fltFreqOffset = GetSingle (&ucData[12]);
    fltTimeOfFix  = GetSingle (&ucData[16]);

    // Format the output string
    m_str.Format ("Vel ENU: %11.3f  %11.3f  %11.3f  %12.3f%s",
                  fltEnuVel[0], fltEnuVel[1], fltEnuVel[2], 
                  fltFreqOffset, ShowTime (fltTimeOfFix));
}

/*-----------------------------------------------------------------------------
Function:       Parse0x6D

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x6D (unsigned char ucData[], int nLen)
{
    U8  ucOperatingMode, ucNumSVs, ucSvPrn[12], ucDimension, i;
    FLT fltPDOP, fltHDOP, fltVDOP, fltTDOP;

    // Extract number of satellites first because the length
    // of the packet depends on the number of reported satellites
    ucNumSVs = (U8)((ucData[0] & 0xF0) >> 4);

    // Check the length of the data string
    if (nLen != 17 + ucNumSVs) 
    {
        return;
    }

    // Extract values from the data string
    ucOperatingMode = (U8)(ucData[0] & 0x08);
    ucDimension     = (U8)(ucData[0] & 0x07);
    fltPDOP         = GetSingle (&ucData[1]);
    fltHDOP         = GetSingle (&ucData[5]);
    fltVDOP         = GetSingle (&ucData[9]);
    fltTDOP         = GetSingle (&ucData[13]);

    for (i=0; i<ucNumSVs; i++) 
    {
        ucSvPrn[i] = ucData[i+17];
    }

    // Format the output string
    m_strTemp.Format ("Mode: %d SV (%d-D, %s):",
                      ucNumSVs, ucDimension - 1, 
                      (ucOperatingMode ? "Manual" : "Auto"));
    m_str += m_strTemp;

    for (i=0; i<ucNumSVs; i++)
    {
        if (ucSvPrn[i]) 
        {
            m_strTemp.Format(" %02d", ucSvPrn[i]);
            m_str += m_strTemp;
        }
    }

    m_strTemp.Format ("\r\n      P, H, V, TDOP = %.2f, %.2f, %.2f, %.2f",
                      fltPDOP, fltHDOP, fltVDOP, fltTDOP);
    m_str += m_strTemp;
}

/*-----------------------------------------------------------------------------
Function:       Parse0x82

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x82 (unsigned char ucData[], int nLen)
{
    U8 ucDiffMode;

    // Check the length of the data string
    if (nLen != 1) 
    {
        return;
    }

    // Extract values from the data string
    ucDiffMode = ucData[0];

    // Format the output string
    m_str.Format ("Fix is%s DGPS-corrected (%s mode)  (%d)",
                  ((ucDiffMode & 1) ? "" : " not"),
                  ((ucDiffMode&2) ? "auto" : "manual"),
                  ucDiffMode);
}

/*-----------------------------------------------------------------------------
Function:       Parse0x83

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x83 (unsigned char ucData[], int nLen)
{
    DBL dblEcefPos[3], dblClockBias;
    FLT fltTimeOfFix;

    // Check the length of the data string
    if (nLen != 36) 
    {
        return;
    }

    // Extract values from the data string
    dblEcefPos[0] = GetDouble (ucData);
    dblEcefPos[1] = GetDouble (&ucData[8]);
    dblEcefPos[2] = GetDouble (&ucData[16]);
    dblClockBias  = GetDouble (&ucData[24]);
    fltTimeOfFix  = GetSingle (&ucData[32]);

    // Format the output string
    m_str.Format ("DXYZ:%12.2f  %13.2f  %13.2f %12.2f%s",
                  dblEcefPos[0], dblEcefPos[1], dblEcefPos[2], 
                  dblClockBias, ShowTime(fltTimeOfFix));
}

/*-----------------------------------------------------------------------------
Function:       Parse0x84

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x84 (unsigned char ucData[], int nLen)
{
    DBL dblLat, dblLon, dblAlt, dblClockBias;
    FLT fltTimeOfFix;
    DBL dblLatMin, dblLonMin;
    S16 sLatDeg, sLonDeg;
    S8  cNorthSouth, cEastWest;

    // Check the length of the data string
    if (nLen != 36) 
    {
        return;
    }

    // Extract values from the data string
    dblLat       = GetDouble (ucData);
    dblLon       = GetDouble (&ucData[8]);
    dblAlt       = GetDouble (&ucData[16]);
    dblClockBias = GetDouble (&ucData[24]);
    fltTimeOfFix = GetSingle (&ucData[32]);

    // Convert from radians to degrees
    dblLat *= R2D;
    dblLon *= R2D;
    if (dblLat < 0.0)
    {
        cNorthSouth = 'S';
        dblLat = -dblLat;
    } 
    else
    {
        cNorthSouth = 'N';
    }
    sLatDeg = (S16)dblLat;
    dblLatMin = (dblLat - sLatDeg) * 60.0;

    if (dblLon < 0.0) 
    {
        cEastWest = 'W';
        dblLon = -dblLon;
    } 
    else
    {
        cEastWest = 'E';
    }
    sLonDeg = (S16)dblLon;
    dblLonMin = (dblLon - sLonDeg) * 60.0;

    // Format the output string
    m_str.Format ("DLLA: %2d:%08.5f %c; %3d:%08.5f %c; %10.2f %12.2f%s",
                  sLatDeg, dblLatMin, cNorthSouth, 
                  sLonDeg, dblLonMin, cEastWest,
                  dblAlt, dblClockBias, ShowTime(fltTimeOfFix));
}

/*-----------------------------------------------------------------------------
Function:       Parse0x8F

Description:    Parses TSIP superpacket 0x8F-xx.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x8F (unsigned char ucData[], int nLen)
{
    // Extract the super-packet identifier contained in the first byte of the
    // TSIP data stream and dispatch to an appropriate parser.
    switch (ucData[0])
    {
        case 0x20: Parse0x8F20 (ucData, nLen); break;
        case 0xAB: Parse0x8FAB (ucData, nLen); break;
        case 0xAC: Parse0x8FAC (ucData, nLen); break;
        default:                               break;
    }
}

/*-----------------------------------------------------------------------------
Function:       Parse0x8F20

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x8F20 (unsigned char ucData[], int nLen)
{
    DBL  dblLat, dblLon, dblAlt, fltTimeOfFix, dblLonDeg, dblLatDeg;
    DBL  dblEnuVel[3], dblVelScale;
    U32  ulTemp;
    S32  lTemp;
    S16  sWeekNum, ucSvIODE[32];
    U8   ucSubpacketID, ucInfo, ucNumSVs, ucSvPrn[32], i, ucPrn, ucMaxSVs;
    S8   cDatumIdx;
    char strDatum[20];

    // Check the length of the data string
    if (nLen == 56)
    {
        ucMaxSVs = 8;
    }
    else if (nLen == 64)
    {
        ucMaxSVs = 12;
    }
    else
    {
        return;
    }

    // Extract values from the data string
    ucSubpacketID = ucData[0];
    dblVelScale   = (ucData[24] & 1) ? 0.020 : 0.005;
    dblEnuVel[0]  = GetShort (&ucData[2]) * dblVelScale;
    dblEnuVel[1]  = GetShort (&ucData[4]) * dblVelScale;
    dblEnuVel[2]  = GetShort (&ucData[6]) * dblVelScale;
    fltTimeOfFix  = GetULong (&ucData[8]) * 0.001;

    lTemp  = GetLong (&ucData[12]);
    dblLat = lTemp*(GPS_PI/MAX_LONG);

    ulTemp = GetULong (&ucData[16]);
    dblLon = ulTemp*(GPS_PI/MAX_LONG);
    if (dblLon > GPS_PI)
    {
        dblLon -= 2.0*GPS_PI;
    }

    dblAlt = GetLong (&ucData[20])*.001;

    /* 25 blank; 29 = UTC */
    cDatumIdx = ucData[26];
    cDatumIdx--;

    ucInfo   = ucData[27];
    ucNumSVs = ucData[28];
    sWeekNum = GetShort (&ucData[30]);

    for (i=0; i<ucMaxSVs; i++) 
    {
        ucPrn       = ucData[32+2*i];
        ucSvPrn[i]  = (U8)(ucPrn & 0x3F);
        ucSvIODE[i] = (U16)(ucData[33+2*i] + 4*((S16)ucPrn-(S16)ucSvPrn[i]));
    }

    // Format the output string
    m_strTemp.Format ("Fix at: %04d:%3s:%02d:%02d:%06.3f GPS (=UTC+%2ds)  FixType: %s%s%s",
                      sWeekNum, gstrDayName[(S16)(fltTimeOfFix/86400.0)],
                      (S16)fmod(fltTimeOfFix/3600., 24.), 
                      (S16)fmod(fltTimeOfFix/60., 60.),
                      fmod(fltTimeOfFix, 60.), 
                      (char)ucData[29],
                      ((ucInfo & INFO_DGPS) ? "Diff" : ""),
                      ((ucInfo & INFO_2D) ? "2D" : "3D"),
                      ((ucInfo & INFO_FILTERED) ? "-Filtrd" : ""));
    m_str += m_strTemp;

    if (cDatumIdx > 0)
    {
        sprintf(strDatum, "Datum%3d", cDatumIdx);
    }
    else if (cDatumIdx)
    {
        sprintf(strDatum, "Unknown ");
    }
    else
    {
        sprintf(strDatum, "WGS-84");
    }

    /* convert from radians to degrees */
    dblLatDeg = R2D * fabs(dblLat);
    dblLonDeg = R2D * fabs(dblLon);

    m_strTemp.Format ("\r\n   Pos: %4d:%09.6f %c %5d:%09.6f %c %10.2f m HAE (%s)",
                      (S16)dblLatDeg, fmod(dblLatDeg, 1.)*60.0, (dblLat<0.0)?'S':'N',
                      (S16)dblLonDeg, fmod(dblLonDeg, 1.)*60.0, (dblLon<0.0)?'W':'E',
                      dblAlt, strDatum);
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n   Vel:    %9.3f E       %9.3f N      %9.3f U   (m/sec)",
                      dblEnuVel[0], dblEnuVel[1], dblEnuVel[2]);
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n   SVs: ");
    m_str += m_strTemp;

    for (i=0; i<ucNumSVs; i++)
    {
        m_strTemp.Format (" %02d", ucSvPrn[i]);
        m_str += m_strTemp;
    }

    m_strTemp.Format ("     (IODEs:");
    m_str += m_strTemp;

    for (i=0; i<ucNumSVs; i++)
    {
        m_strTemp.Format (" %02X", ucSvIODE[i] & 0xFF);
        m_str += m_strTemp;
    }

    m_strTemp.Format(")");
    m_str += m_strTemp;
}

/*-----------------------------------------------------------------------------
Function:       Parse0x8FAB

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x8FAB (unsigned char ucData[], int nLen)
{
    U32 ulTimeOfWeek;
    U16 usWeekNumber, usYear;
    S16 sUtcOffset;
    U8  ucTimingFlag, ucSecond, ucMinute, ucHour, ucDay, ucMonth;    

    // Check the length of the data string
    if (nLen != 17) 
    {
        return;
    }

    // Extract values from the data string
    ulTimeOfWeek = GetULong (&ucData[1]);
    usWeekNumber = GetUShort (&ucData[5]);
    sUtcOffset   = GetShort (&ucData[7]);
    ucTimingFlag = ucData[9];
    ucSecond     = ucData[10];
    ucMinute     = ucData[11];
    ucHour       = ucData[12];
    ucDay        = ucData[13];
    ucMonth      = ucData[14];
    usYear       = GetUShort (&ucData[15]);

    // Format the output string
    m_strTemp.Format ("8FAB: TOW: %06d  WN: %04d", ulTimeOfWeek, usWeekNumber);
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n      %04d/%02d/%02d  %02d:%02d:%02d",
                      usYear, ucMonth, ucDay, ucHour, ucMinute, ucSecond);
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n      UTC Offset: %d s   Timing flag: 000%d%d%d%d%d",
                      sUtcOffset,
                      ((ucTimingFlag>>4) & 1),
                      ((ucTimingFlag>>3) & 1),
                      ((ucTimingFlag>>2) & 1),
                      ((ucTimingFlag>>1) & 1),
                      ((ucTimingFlag   ) & 1));
    m_str += m_strTemp;
}

/*-----------------------------------------------------------------------------
Function:       Parse0x8FAC

Description:    Extracts the data values from the TSIP packet and fills the
                global m_str variable with ASCII representations of the data
                values.

Parameters:     ucData - a pointer to the start of the TSIP data values buffer
                nLen   - number of TSIP data bytes in the data buffer ucData

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipParser::Parse0x8FAC (unsigned char ucData[], int nLen)
{
    U8  ucReceiverMode, ucDiscipliningMode, ucSelfSurveyProgress;
    U8  ucGPSDecodingStatus, ucDiscipliningActivity,ucSpareStatus1;
    U8  ucSpareStatus2;
    U16 usCriticalAlarms, usMinorAlarms;
    FLT fltPPSQuality, fltTenMHzQuality, fltDACVoltage, fltTemperature;
    U32 ulHoldoverDuration, ulDACValue;
    DBL dblLatitude, dblLongitude, dblAltitude, dblLatDeg, dblLonDeg;

    // Check the length of the data string
    if (nLen != 68) 
    {
        return;
    }

    // Extract values from the data string
    ucReceiverMode         = ucData[1];
    ucDiscipliningMode     = ucData[2];
    ucSelfSurveyProgress   = ucData[3];
    ulHoldoverDuration     = GetULong(&ucData[4]);
    usCriticalAlarms       = GetUShort(&ucData[8]);
    usMinorAlarms          = GetUShort(&ucData[10]);
    ucGPSDecodingStatus    = ucData[12];
    ucDiscipliningActivity = ucData[13];
    ucSpareStatus1         = ucData[14];
    ucSpareStatus2         = ucData[15];
    fltPPSQuality          = GetSingle(&ucData[16]);
    fltTenMHzQuality       = GetSingle(&ucData[20]);
    ulDACValue             = GetULong(&ucData[24]);
    fltDACVoltage          = GetSingle(&ucData[28]);
    fltTemperature         = GetSingle(&ucData[32]);
    dblLatitude            = GetDouble(&ucData[36]);
    dblLongitude           = GetDouble(&ucData[44]);
    dblAltitude            = GetDouble(&ucData[52]);

    // These text descriptions are used for formatting below.
    const char *strOprtngDim[8] = 
    {
        "Automatic (2D/3D)",
        "Single Satellite (Time)",
        "unknown",
        "Horizontal (2D)",
        "Full Position (3D)",
        "DGPR Reference",
        "Clock Hold (2D)",
        "Overdetermined Clock"
    };

    // Format the output string
    m_strTemp.Format ("8FAC: RecvMode: %s   DiscMode: %d   SelfSurv: %d   Holdover: %d s",
                      strOprtngDim[ucReceiverMode%7], ucDiscipliningMode, 
                      ucSelfSurveyProgress, ulHoldoverDuration);
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n      Crit: %d%d%d%d.%d%d%d%d   Minr: %d%d%d%d.%d%d%d%d",
                      ((usCriticalAlarms >> 7) & 1),
                      ((usCriticalAlarms >> 6) & 1),
                      ((usCriticalAlarms >> 5) & 1),
                      ((usCriticalAlarms >> 4) & 1),
                      ((usCriticalAlarms >> 3) & 1),
                      ((usCriticalAlarms >> 7) & 1),
                      ((usCriticalAlarms >> 1) & 1),
                      ((usCriticalAlarms     ) & 1),
                      ((usMinorAlarms >> 7) & 1),
                      ((usMinorAlarms >> 6) & 1),
                      ((usMinorAlarms >> 5) & 1),
                      ((usMinorAlarms >> 4) & 1),
                      ((usMinorAlarms >> 3) & 1),
                      ((usMinorAlarms >> 7) & 1),
                      ((usMinorAlarms >> 1) & 1),
                      ((usMinorAlarms     ) & 1));
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n      GPS Status: %d   Discpln Act: %d   Spare Status: %d %d",
                      ucGPSDecodingStatus, ucDiscipliningActivity, 
                      ucSpareStatus1, ucSpareStatus2);
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n      Qual:  PPS: %.1f ns   Freq: %.3f PPB",
                      fltPPSQuality, fltTenMHzQuality);
    m_str += m_strTemp;

    m_strTemp.Format ("\r\n      DAC:  Value: %d   Voltage: %f   Temp: %f deg C",
                      ulDACValue, fltDACVoltage, fltTemperature);
    m_str += m_strTemp;

    /* convert from radians to degrees */
    dblLatDeg = R2D * fabs(dblLatitude);
    dblLonDeg = R2D * fabs(dblLongitude);

    m_strTemp.Format ("\r\n      Pos:  %d:%09.6f %c   %d:%09.6f %c   %.2f m ",
                      (short)dblLatDeg, fmod (dblLatDeg, 1.)*60.0,
                      (dblLatitude<0.0)?'S':'N',
                      (short)dblLonDeg, fmod (dblLonDeg, 1.)*60.0,
                      (dblLongitude<0.0)?'W':'E',
                      dblAltitude);    
    m_str += m_strTemp;
}

/*---------------------------------------------------------------------------*\
 |            D A T A   V A L U E   E X T R A C T   R O U T I N E S
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       GetShort

Description:    Extracts an S16 value from the data buffer pointed to by pucBuf.

Parameters:     pucBuf - a pointer to the data from which to extract the value

Return Value:   Extracted value
-----------------------------------------------------------------------------*/
S16 CTsipParser::GetShort (U8 *pucBuf)
{
    U8 ucBuf[2];

    ucBuf[0] = pucBuf[1];
    ucBuf[1] = pucBuf[0];

    return (*((S16*)ucBuf));
}

/*-----------------------------------------------------------------------------
Function:       GetUShort

Description:    Extracts a U16 value from the data buffer pointed to by pucBuf.

Parameters:     pucBuf - a pointer to the data from which to extract the value

Return Value:   Extracted value
-----------------------------------------------------------------------------*/
U16 CTsipParser::GetUShort (U8 *pucBuf)
{
    U8 ucBuf[2];

    ucBuf[0] = pucBuf[1];
    ucBuf[1] = pucBuf[0];

    return (*((U16*)ucBuf));
}

/*-----------------------------------------------------------------------------
Function:       GetLong

Description:    Extracts an S32 value from the data buffer pointed to by pucBuf.

Parameters:     pucBuf - a pointer to the data from which to extract the value

Return Value:   Extracted value
-----------------------------------------------------------------------------*/
S32 CTsipParser::GetLong (U8 *pucBuf)
{
    U8 ucBuf[4];

    ucBuf[0] = pucBuf[3];
    ucBuf[1] = pucBuf[2];
    ucBuf[2] = pucBuf[1];
    ucBuf[3] = pucBuf[0];

    return (*((S32*)ucBuf));
}

/*-----------------------------------------------------------------------------
Function:       GetULong

Description:    Extracts a U32 value from the data buffer pointed to by pucBuf.

Parameters:     pucBuf - a pointer to the data from which to extract the value

Return Value:   Extracted value
-----------------------------------------------------------------------------*/
U32 CTsipParser::GetULong (U8 *pucBuf)
{
    U8 ucBuf[4];

    ucBuf[0] = pucBuf[3];
    ucBuf[1] = pucBuf[2];
    ucBuf[2] = pucBuf[1];
    ucBuf[3] = pucBuf[0];

    return (*((U32 *)ucBuf));
}

/*-----------------------------------------------------------------------------
Function:       GetSingle

Description:    Extracts a FLT value from the data buffer pointed to by pucBuf.

Parameters:     pucBuf - a pointer to the data from which to extract the value

Return Value:   Extracted value
-----------------------------------------------------------------------------*/
FLT CTsipParser::GetSingle (U8 *pucBuf)
{
    U8 ucBuf[4];

    ucBuf[0] = pucBuf[3];
    ucBuf[1] = pucBuf[2];
    ucBuf[2] = pucBuf[1];
    ucBuf[3] = pucBuf[0];

    return (*((FLT *)ucBuf));
}

/*-----------------------------------------------------------------------------
Function:       GetDouble

Description:    Extracts a DBL value from the data buffer pointed to by pucBuf.

Parameters:     pucBuf - a pointer to the data from which to extract the value

Return Value:   Extracted value
-----------------------------------------------------------------------------*/
DBL CTsipParser::GetDouble (U8* pucBuf)
{
    U8 ucBuf[8];

    ucBuf[0] = pucBuf[7];
    ucBuf[1] = pucBuf[6];
    ucBuf[2] = pucBuf[5];
    ucBuf[3] = pucBuf[4];
    ucBuf[4] = pucBuf[3];
    ucBuf[5] = pucBuf[2];
    ucBuf[6] = pucBuf[1];
    ucBuf[7] = pucBuf[0];

    return (*((DBL *)ucBuf));
}

/*---------------------------------------------------------------------------*\
 |                       H E L P E R   R O U T I N E S
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       TsipShowTime

Description:    Convert time of week into day-hour-minute-second and print

Parameters:     fltTimeOfWeek - time of week

Return Value:   A pointer to the formatted time string
-----------------------------------------------------------------------------*/
CString CTsipParser::ShowTime (FLT fltTimeOfWeek)
{
    S16     sDay, sHour, sMinute;
    FLT     fltSecond;
    DBL     dblTimeOfWeek;
    CString strTime;

    if (fltTimeOfWeek == -1.0)
    {
        strTime = "   <No time yet>   ";
    }
    else if ((fltTimeOfWeek >= 604800.0) || (fltTimeOfWeek < 0.0))
    {
        strTime = "     <Bad time>     ";
    }
    else
    {
        if (fltTimeOfWeek < 604799.9) 
        {
            dblTimeOfWeek = fltTimeOfWeek + .00000001;
        }

        fltSecond = (FLT)fmod(dblTimeOfWeek, 60.);
        sMinute   =  (S16) fmod(dblTimeOfWeek/60., 60.);
        sHour     = (S16)fmod(dblTimeOfWeek / 3600., 24.);
        sDay      = (S16)(dblTimeOfWeek / 86400.0);

        strTime.Format (" %s %02d:%02d:%05.2f   ",
                        gstrDayName[sDay], sHour, sMinute, fltSecond);
    }

    return strTime;
}
